' Mancala Game - 1 player against computer
' Rev 1.0.0 William M Leue 6/12/2022

option default integer
option base 1
option angle degrees


const NUM_PITS = 6
const NUM_STONES = 4
const TOTAL_STONES = 2*NUM_PITS*NUM_STONES
const NUM_PIT_SLOTS = 24
const NUM_MANCALA_SLOTS = 48

const BOARD_WIDTH = 720
const BOARD_HEIGHT = 300
const PIT_RAD = 35
const GAP = 15
const PIT_SPACING = 2*PIT_RAD + GAP
const MANCALA_HEIGHT = 200
const STONE_RAD = 5
const RLEN = 2*NUM_PITS+2
const MANCALA1 = 7
const MANCALA2 = 14

const MX1 = MM.HRES\2 + BOARD_WIDTH\2-GAP-PIT_RAD
const MX2 = MM.HRES\2-BOARD_WIDTH\2+GAP+PIT_RAD

const BCOLOR = RGB(150, 70, 20)
const PCOLOR = RGB(100, 35, 10)
const RCOLOR = RGB(BLACK)

const SRED = RGB(RED)
const SGREEN = RGB(GREEN)
const SBLUE = RGB(BLUE)
const SYELLOW = RGB(YELLOW)

const USER = 1
const COMPUTER = 2

const LEFT  = 130
const RIGHT = 131
const ENTER = 13
const ESC   = 27
const QUEST = 63

const SLINE_XOFF = PIT_RAD+20
const SLINE_YOFF = PIT_RAD+20

const STATS_W = 400
const STATS_H = 200

' Globals
dim pit_nstones(RLEN)
dim stones(TOTAL_STONES, 3)
dim pit_slots(RLEN, NUM_MANCALA_SLOTS)
dim pit_locs(RLEN, 2)
dim stoneColor(4) = (SRED, SGREEN, SBLUE, SYELLOW)
dim round(RLEN) = (11, 12, 13, 14, 15, 16, 17, 21, 22, 23, 24, 24, 25, 27)
dim firstMover = COMPUTER
dim currentMover
dim userChoice
dim computerChoice
dim currentChoiceEndPit = 0
dim pit_X_coords(RLEN) = (188,273,358,443,528,613,MX1,613,528,443,358,273,188,MX2)
dim pit_Y_coords(2)    = (400, 200)
dim mancala_slot_locs(2, 50, 2)
dim lastDestinationPit
dim sowing_locs(RLEN, 3)
dim user_pit_indices(NUM_PITS) = (1, 2, 3, 4, 5, 6)
dim cptr_pit_indices(NUM_PITS) = (8, 9, 10, 11, 12, 13)
dim num_usermoves = 0
dim num_computermoves = 0
dim capturePit = 0
dim num_games = 0
dim num_user_wins = 0
dim num_ties = 0
dim running = 0

' Main Program
open "debug.txt" for output as #1
cls RGB(white)
MakePitLocations
MakeMancalaSlotLocs
do
  NewGame
  DrawBoard
  GameEventLoop
  ShowGameStats
loop
end

' Make the locations of the centers of the user and computer pits
' and Mancalas.
sub MakePitLocations
  local i
  for i = 1 to RLEN
    pit_locs(i, 1) = pit_X_coords(i)
    if i <= NUM_PITS then
      pit_locs(i, 2) = pit_Y_coords(1)
    else if i = NUM_PITS+1 then
      pit_locs(i, 2) = MM.VRES\2
    else if i <= 2*NUM_PITS+1 then
      pit_locs(i, 2) = pit_Y_coords(2)
    else
      pit_locs(i, 2) = MM.VRES\2
    end if
  next i
end sub

' Make the stone locations in the mancalas
sub MakeMancalaSlotLocs 
  local i, j, k, n, mx(2), my
  local xlocs(5), ylocs(10)
  for i = 1 to 5
    read xlocs(i)
  next i
  for i = 1 to 10
    read ylocs(i)
  next i
  mx(1) = MM.HRES\2 + BOARD_WIDTH\2-2*PIT_RAD-GAP
  mx(2) = MM.HRES\2 - BOARD_WIDTH\2 + GAP
  for i = 1 to 2
    n = 0
    for j = 1 to 10
      for k = 1 to 5
        inc n
        mancala_slot_locs(i, n, 1) = mx(i) + xlocs(k)
        mancala_slot_locs(i, n, 2) = ylocs(j)
      next k
    next j
  next i
end sub

' Draw the Mancala Board in Page 1
sub DrawBoard
  local i, x, y, bw, gh, mh, pr, pd, g, pc
  local uchoice, cchoice
  bw = BOARD_WIDTH
  x = MM.HRES\2
  y = MM.VRES\2
  bh = BOARD_HEIGHT
  mh = MANCALA_HEIGHT
  pr = PIT_RAD
  pd = 2*PIT_RAD
  g = GAP

  page write 1
  cls RGB(WHITE)
  text MM.HRES\2, 10, "Mancala", "CT", 5,, RGB(BLUE), RGB(WHITE)
  text MM.HRES\2, 40, "Press '?' for Help", "CT",,, RGB(BLACK), -1
  cc = RGB(BLACK)
  uc = RGB(BLACK)
  if currentChoiceEndPit = MANCALA1 then
    cc = RGB(BLUE)
  else if currentChoiceEndPit = MANCALA2 then
    uc = RGB(BLUE)
  end if
  rbox x-bw\2, y-bh\2, bw, bh,, BCOLOR, BCOLOR
  rbox x-bw\2+g, y-mh\2, pd, mh, g, uc, PCOLOR
  rbox x+bw\2-pd-g, y-mh\2, pd, mh, g, cc, PCOLOR
  for i = 1 to RLEN
    if i <> MANCALA1 and i <> MANCALA2 then
      x = pit_locs(i, 1) : y = pit_locs(i, 2)
      if currentMover = USER then
        if i = userChoice then
          pc= RGB(YELLOW)
        else
          pc = RGB(BLACK)
        end if
      else if currentMover = COMPUTER then
        if i = computerChoice then
          pc= RGB(YELLOW)
        else
          pc = RGB(BLACK)
        end if
      end if
      if i = currentChoiceEndPit then
        pc = RGB(BLUE)
      end if
      if i = capturePit then
        pc = RGB(RED)
      end if
      circle x, y, pr,,, pc, PCOLOR
    end if
  next i
  page write 0
  page copy 1 to 0, B
  DrawStones
end sub

' Draw all the stones on the board
sub DrawStones
  local i, where, slot, c, x, y, rx, ry
  for i = 1 to TOTAL_STONES
    where = stones(i, 1)
    slot = stones(i, 2)
    c = stoneColor(stones(i, 3))
    x = pit_locs(where, 1) : y = pit_locs(where, 2)
    GetSlotOffsets where, x, y, slot, rx, ry
    circle rx, ry, STONE_RAD,,, RGB(BLACK), c
  next i      
  LabelPits
end sub

' Label the pits with the number of stones
sub LabelPits
  local i, n, x, y, py
  for i = 1 to RLEN
    n = pit_nstones(i)
    x = pit_X_coords(i)
    if i <= NUM_PITS+1 then
      if i = MANCALA1 then
        y = MM.VRES\2 - MANCALA_HEIGHT\2-10
      else
        y = pit_Y_coords(1) + PIT_RAD + 6
      end if
    else
      if i = MANCALA2 then
        y = MM.VRES\2 - MANCALA_HEIGHT\2-10
      else
        y = pit_Y_coords(2) - PIT_RAD - 10
      end if
    end if
    text x, y, "   ", "CT", 7,,, BCOLOR
    text x, y, str$(n), "CT", 7,,, BCOLOR
  next i
end sub    

' Set up a new Game
sub NewGame
  local i, j, n
  running = 0
  for i = 1 to RLEN
    pit_nstones(i) = 0
    for j = 1 to NUM_MANCALA_SLOTS
      pit_slots(i, j) = 0
    next j
  next i
  n = 1
  for i = 1 to RLEN
    if i <> MANCALA1 and i <> MANCALA2 then
      for j = 1 to NUM_STONES
        AddStoneToPit 0, i, n, j
        inc n
      next j
    end if
  next i
  if firstMover = USER then
    firstMover = COMPUTER
  else
    firstMover = USER
  end if
  currentMover = firstMover
  userChoice = user_pit_indices(1)
  lastDestinationPit = 0
  num_usermoves = 0
  num_computermoves = 0
  running = 1
end sub

' Show Game Statistics, ask if user wants to play again
sub ShowGameStats
  local x, y, m$, z$, ncw
  x = mm.hres\2 - STATS_W\2
  y = mm.vres\2 - STATS_H\2
  box x, y, STATS_W, STATS_H,, rgb(black), rgb(green)
  ncw = num_games - (num_user_wins + num_ties)
  text x+STATS_W\2, y+10, "Game Statistics", "CT", 4,, rgb(black), -1
  m$ = "Number of Games played: " + str$(num_games)
  text x+10, y+30, m$, "LT",,, rgb(black), -1
  m$ = "Number of User wins: " + str$(num_user_wins)
  text x+10, y+45, m$, "LT",,, rgb(black), -1
  m$ = "Number of Computer wins: " + str$(ncw)
  text x+10, y+60, m$, "LT",,, rgb(black), -1
  m$ = "Number of Tie Games: " + str$(num_ties)
  text x+10, y+75, m$, "LT",,, rgb(black), -1
  text x+STATS_W\2, y+180, "Press 'Y' to play again or Escape to Quit", "CT",,, rgb(black), -1
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  if UCASE$(z$) <> "Y" then Quit
end sub

' Add a stone to a pit
' pit_index is in global context
sub AddStoneToPit who, pit_index, stone_index, cindex
  if pit_index = 0 or pit_index > RLEN then
    ERROR "BUG - pit index " + str$(pit_index) + " out of range"
  end if
  inc pit_nstones(pit_index)
  slot = GetFreeSlot(pit_index)
  pit_slots(pit_index, slot) = 1
  stones(stone_index, 1) = pit_index
  stones(stone_index, 2) = slot
  stones(stone_index, 3) = cindex
  lastDestinationPit = pit_index
  on error skip 1
  if running then play WAV "Stone1.wav"
end sub

' Get a free slot in a pit
' pit_index is in global context
function GetFreeSlot(pit_index)
  local slot, n
  if pit_index = MANCALA1 or pit_index = MANCALA2 then
    n = 0
    do
      inc n
      if n > 100 then
        ERROR "Bug: too many free slot tries (mancala)"
      end if
      slot = GetRandomIntegerInRange(1, NUM_MANCALA_SLOTS)
      if pit_slots(pit_index, slot) = 0 then exit do
    loop
  else
    n = 0
    do
      inc n
      if n > 100 then
        ERROR "Bug: too many free slot fries (normal)"
      end if
      slot = GetRandomIntegerInRange(1, NUM_PIT_SLOTS)
      if pit_slots(pit_index, slot) = 0 then exit do
    loop
  end if
  GetFreeSlot = slot
end function

' Convert a slot index into a position in a pit
sub GetSlotOffsets where, x, y, slot, rx, ry
  local float r, a, ainc, y1
  local p
  if where <> MANCALA1 and where <> MANCALA2 then
    if slot < 12 then
      r = 3.0*PIT_RAD/4.0
      a = slot
      ainc = 30.0
    else if slot < 20 then
      r = 2.0*PIT_RAD/4.0
      a = slot - 12
      ainc = 45.0
    else
      r = PIT_RAD/4.0
      a = slot - 20
      ainc = 90.0
    end if
    y1 = y  
    rx = x + int(r*cos(a*ainc) + 0.5)
    ry = y1 - int(r*sin(a*ainc) + 0.5)
  else
    if where = MANCALA1 then p = 1
    if where = MANCALA2 then p = 2
    rx = mancala_slot_locs(p, slot, 1)
    ry = mancala_slot_locs(p, slot, 2)
  end if
end sub

' Do a Move
' Who: USER = 1, COMPUTER = 2
' which: mover's chosen pit index
sub Move who, which
  local i, n, ix, fx, k, sn, c, slot, p
  local fstones(TOTAL_STONES)
  n = pit_nstones(which)
  if n = 0 then
    ERROR "BUG - no stones in move pit " + str$(which)
    exit sub
  end if
  pit_nstones(which) = 0
  FindStones which, fstones()
  k = 0
  for i = which+1 to which+n
    inc k
    sn = fstones(k)
    if sn = 0 then ERROR "BUG: stone number zero in Move"
    c = stones(sn, 3)
    slot = stones(sn, 2)
    pit_slots(which, slot) = 0
    p = (i-1) mod RLEN + 1
    AddStoneToPit who, p, sn, c
    DrawBoard
    pause 400
  next i  
  CheckCapture who
end sub

' Given a pit, return a list of stones there
' gwhere is in global context.
sub FindStones gwhere, fstones()
  local n = 0
  for i = 1 to TOTAL_STONES
    if stones(i, 1) = gwhere then
      inc n
      fstones(n) = i
    end if
  next i
end sub    

' Game Event Loop (no controller)
sub GameEventLoop
  local cmd
  local z$
  do
    if CheckGameOver() then exit do
    if currentMover = USER then
      ShowMoveEnd(USER, userChoice)
      computerChoice = 0
      DrawBoard
      Prompt("Your Move")
      z$ = INKEY$
      do
        z$ = INKEY$
      loop until z$ <> ""
      cmd = asc(z$)
      select case cmd
        case ESC
          Quit
        case QUEST
          ShowHelp
          DrawBoard
        case LEFT
            if userChoice > user_pit_indices(1) then
            inc userChoice, -1
          else
            userChoice = user_pit_indices(NUM_PITS)
          end if
            ShowMoveEnd(USER, userChoice)
        case RIGHT
          if userChoice < user_pit_indices(NUM_PITS) then
            inc userChoice
          else
            userChoice = user_pit_indices(1)
          end if
            ShowMoveEnd(USER, userChoice)
        case ENTER
          if pit_nstones(userChoice) = 0 then
            print #1, "No stones in pit " + str$(userChoice)
          end if
          if pit_nstones(userChoice) > 0 then
            inc num_usermoves
            Move USER, userChoice            
            CurrentChoiceEndPit = 0
            DrawBoard
            if CheckGameOver() then exit do
            extra = IsExtraMove(USER)
            if extra = 1 then
              Prompt "You get an Extra Move"
              pause 1000
            else
              currentMover = COMPUTER
            end if
            if CheckGameOver() then exit do
          end if
        case else
      end select
    else
      userChoice = 0
      DrawBoard
      Prompt "Computer Thinking..."
      pause 1000
      computerChoice = GetComputerMove()
      DrawBoard
      ShowMoveEnd(COMPUTER, computerChoice)
      pause 1000
      inc num_computermoves
      Move COMPUTER, computerChoice
      currentChoiceEndPit = 0
      if CheckGameOver() then exit do
      if IsExtraMove(COMPUTER) = 1 then
        Prompt "Computer Gets an Extra Move"
        pause 1000
        Prompt "Computer Thinking..."
      else
        currentMover = USER
      end if
      userChoice = 1
      DrawBoard
      if CheckGameOver() then exit do
    end if
  loop
end sub

' Terminate the program
sub Quit
  close #1
  cls
  end
end sub

' If the last stone sown is to the mover's
' mancala, then the mover gets an extra move
function IsExtraMove(who)
  local extra = 0
  if capturePit > 0 then
    IsExtraMove = 0
    capturePit = 0
    exit function
  end if
  if who = USER then
    if lastDestinationPit = 7 then
      extra = 1
    end if
  else
    if lastDestinationPit = 14 then
      extra = 1
    end if
  end if
  IsExtraMove = extra
  lastDestinationPit = 0
end function

' Draw a marker on the ending pit of a proposed
' move, who = player moving, which = start pit index.
sub ShowMoveEnd who, which
  local ix, n, p, x, y
  n = pit_nstones(which)
  p = (which+n-1) mod RLEN + 1
  if p <> which then
    currentChoiceEndPit = p
  end if
end sub

' If the last stone sown on a move is to
' his own empty pit, then the player can capture
' all the stones in the opposite pit if there
' are any. 'who' is the player who moved.
sub CheckCapture who
  local n, t, p
  local fstones(TOTAL_STONES)
  p = lastDestinationPit
  n = pit_nstones(p)
  if n = 1 then
    if who = USER then
      if p >= 1 and p <= NUM_PITS then
        t = RLEN - p
        n = pit_nstones(t)
        if n > 0 then
          capturePit = t
          DrawBoard
          pause 1000
          Capture who, t, p
        end if
      end if
    else
      if p >= NUM_PITS+2 and p < RLEN then
        t = RLEN - p
        n = pit_nstones(t)
        if n > 0 then
          capturePit = t
          DrawBoard
          pause 1000
          Capture who, t, p
        end if
      end if
    end if
  end if
end sub

' Capture Stones for player 'who' from pit 'where'.
sub Capture who, where, p
  local i, n, sn, c, psn
  local fstones(TOTAL_STONES)
  if who = USER then
    n = pit_nstones(where)
    FindStones where, fstones()
    if n = 0 then exit sub
    for i = 1 to n
      sn = fstones(i)
      c = stones(sn, 3)
      AddStoneToPit who, MANCALA1, sn, c
      pause 400
      DrawBoard
    next i
    pause 400
    FindStones p, fstones()
    sn = fstones(1)
    c = stones(sn, 3)
    AddStoneToPit who, MANCALA1, fstones(1), c
    pit_nstones(p) = 0
    pit_nstones(where) = 0
  else
    n = pit_nstones(where)
    FindStones  where, fstones()
    if n = 0 then exit sub
    for i = 1 to n
      sn = fstones(i)
      c = stones(sn, 3)
      AddStoneToPit who, MANCALA2, sn, c
      pause 400
      DrawBoard
    next i
    pause 400
    FindStones p, fstones()
    sn = fstones(1)
    c = stones(sn, 3)
    AddStoneToPit who, MANCALA2, fstones(1), c
    pit_nstones(p) = 0
    pit_nstones(where) = 0
  end if
end sub

' See if one user has emptied their pits
' If so, the game is over. Count up stones in the
' players' mancalas plus leftover stones to determine winner.
function CheckGameOver()
  local i, j, k, n, sn
  local over = 0
  local cnt = 0
  local uclr = 0
  local cclr = 0
  local fstones(TOTAL_STONES)

  for i = user_pit_indices(1) to user_pit_indices(NUM_PITS)
    inc cnt, pit_nstones(i)
  next i
  if cnt = 0 then 
    over = 1
    uclr = 1
  else
    cnt = 0
    for i = cptr_pit_indices(1) to cptr_pit_indices(NUM_PITS)
      inc cnt, pit_nstones(i)
    next i
    if cnt = 0 then
      over = 1
      cclr = 1
    end if
  end if
  if over then
    running = 0
    inc num_games
    unum = pit_nstones(MANCALA1)
    cnum = pit_nstones(MANCALA2)
    if uclr then
      for i = NUM_PITS+2 to RLEN-1
        n = pit_nstones(i)
        if n > 0 then
          FindStones i, fstones()
          k = 0
          for j = 1 to n
            inc k
            sn = fstones(k)
            c = stones(sn, 3)
            AddStoneToPit 0, MANCALA1, sn, c
          next j
          unum = unum + n
        end if
      next i
    else
      for i = 1 to NUM_PITS
        n = pit_nstones(i)
        if n > 0 then
          FindStones i, fstones()
          k = 0
          for j = 1 to n
            inc k
            sn = fstones(k)
            c = stones(sn, 3)
            AddStoneToPit 0, MANCALA2, sn, c
          next j
          cnum = cnum + n
        end if
      next i
    end if
    DrawBoard
    if unum > cnum then
      Prompt "You Win, " + str$(unum) + " to " + str$(cnum) + "!"
      inc num_user_wins
    else if cnum > unum then
      Prompt "Computer Wins, " + str$(cnum) + " to " + str$(unum) + "!"
    else
      inc num_ties
      Prompt "A Tie Game, " + str$(unum) + " to " + str$(cnum) + "!"
    end if
    pause 3000
  end if
  CheckGameOver = over
end function

' Get Computer Move
' (no lookahead yet)
function GetComputerMove()
  local i, n, m
  local emoves(NUM_PITS)
  m = 0
  FindExtraMoves COMPUTER, n, emoves()
  if n > 0 then
    m = emoves(1)
  else
    m = FindCaptureMoves(COMPUTER)
    if m = 0 then
      m = GetLargestPit(COMPUTER)
    end if
  end if
  if m = 0 then ERROR "BUG - Computer move zero!"
  GetComputerMove = m
end function

' Looks for and makes a list of extra moves for
' the designated player.
sub FindExtraMoves who, n, emoves()
  local i, ix, nstones, last_pit
  n = 0
  for i = NUM_PITS to 1 step -1
    if who = USER then
      ix = user_pit_indices(i)
    else
      ix = cptr_pit_indices(i)
    end if
    nstones = pit_nstones(ix)
    last_pit = (ix+nstones-1) mod RLEN + 1
    if who = 1 and last_pit = MANCALA1 then
      inc n
      emoves(n) = ix
    else if who = 2 and last_pit = MANCALA2 then
      inc n
      emoves(n) = ix
    end if
  next i
end sub

' Looks for opportunities to capture player's stones and returns
' the one that captures the most stones, or zero if none are possible.
function FindCaptureMoves(who)
  local i, ix, nstones, last_pit, lstones, cap_px, cap_ns
  cap_px = 0
  cap_ns = 0
  for i = NUM_PITS to 1 step -1
    if who = USER then
      ix = user_pit_indices(i)
    else
      ix = cptr_pit_indices(i)
    end if
    nstones = pit_nstones(ix)
    last_pit = (ix+nstones-1) mod RLEN + 1
    if last_pit <> ix then
      lstones = pit_nstones(last_pit)
      if lstones = 0 then
        t = RLEN - ix
        tstones = pit_nstones(t)
        if tstones > cap_ns then
          cap_ns = tstones
          cap_px = ix
        end if
      end if
    end if
  next i
  FindCaptureMoves = cap_px
end function

' returns the pit index for the pit that has
' the largest number of stones on the designated
' player's side.
function GetLargestPit(who)
  local i, n, ix
  local nmax = 0
  local m = 1
  for i = 1 to NUM_PITS
    if who = USER then
      ix = user_pit_indices(i)
    else
      ix = cptr_pit_indices(i)
    end if
    n = pit_nstones(ix)
    if n > nmax then 
      nmax = n
      m = ix
    end if
  next i
  if nmax = 0 then
    ERROR "BUG -- largest pit has zero stones"
  end if
  GetLargestPit = m
end function

' Write a Prompt Text
sub Prompt msg$
  text 50, MM.VRES\2+BOARD_HEIGHT\2+20, space$(50),,,, RGB(BLACK), RGB(WHITE)
  text 50, MM.VRES\2+BOARD_HEIGHT\2+20, msg$,,,, RGB(BLACK), RGB(WHITE)
end sub
        
' Return a random integer in a specified closed range
function GetRandomIntegerInRange(low, high)
  local r
  do
    r = int(low + rnd()*(high-low+1))
    if r >= low and r <= high then exit do
  loop
  GetRandomIntegerInRange = r
end function

' Explain the game
sub ShowHelp
  local z$, nlines, i, y, h$
  cls RGB(WHITE)
  text mm.hres\2, 20, "Mancala Rules and Game Play", "CT", 4,, rgb(blue), -1
  read nlines
  y = 40
  for i = 1 to nlines
    read h$
    print @(20, y, 2) h$
    inc y, 15
  next i
  print @(300, 580, 2) "Press any key to continue"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  cls RGB(WHITE)
  text mm.hres\2, 20, "Graphical Hints", "CT", 4,, rgb(blue), -1
  read nlines
  y = 40
  for i = 1 to nlines
    read h$
    print @(20, y, 2) h$
    inc y, 15
  next i
  print @(300, 580, 2) "Press any key to continue"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub

' Data for Mancala Slots
data  10,  22,  34,  46,  58
data 218, 236, 254, 272, 290
data 308, 326, 344, 362, 380

data 35
data "Mancala is a game of stone moving where the object is to end up with the most"
data "stones. You have 6 'pits' at the bottom of the board; your opponent's pits are"
data "opposite at the top of the board. Each pit starts with 4 stones. To your right"
data "is your Mancala, the large oval pit. This is where you try to accumulate stones,"
data "and your opponent tries to accumulate stones in the left pit. At the end of a"
data "game, the player whose Mancala has the most stones wins."
data ""
data "A MOVE in Mancala consits of picking up ALL the stones in one of your 6 pits, and"
data "then distributing them into the pits that run COUNTER-CLOCKWISE around the board starting"
data "from the pit immediately to the right of the pit where you picked up the stones."
data "Deposit one stone in each pit until your hand is empty. If there are a lot of"
data "stones in the starting pit you choose, your move may run completely around the"
data "board and end up putting stones into your own pits. Both your and your opponent's"
data "Mancalas count as pits."
data ""
data "Each stone that ends up in your Mancala, whether you put it there or your opponent"
data "did, it counts as one point in your score. The same is true for your opponent."
data ""
data "There are two SPECIAL rules that affect game play. The first is the EXTRA MOVE"
data "rule: If your move ends so that the last pit you put a stone into is your Mancala,"
data "then you get an extra move that you take immediately. With clever play, you may be"
data "able to make several moves in a row before your opponent can move again."
data ""
data "The second special rule is that if your move ends at one of your 6 pits (not the"
data "Mancala) and that pit was empty before you put the last stone into it, then you can"
data "capture all of the stones in the opponent's pit that is directly across the board"
data "from your pit. All these captured stones go directly into your own Mancala, as does"
data "the last stone you dropped. This also ends your move.
data ""
data "Game Commands:"
data "     Starting a move: use the LEFT and RIGHT arrow keys on the keyboard to move"
data "     back and forth between your 6 pits. The currently chosen pit will be hilighted"
data "     in YELLOW. Press ENTER to select that pit and do the move."
data "     To repeat this Help screen, press '?'"
data "     To quit the game, press the Escape key."

data 17
data "The white number near each pit shows the number of stones in that pit.
data ""
data "When you are choosing a move, the game outlines one of the pits with a BLUE outline"
data "to show you where the last stone will be dropped if you perform the move indicated"
data "by the pit with the YELLOW outline. This can help you find a move that ends in your"
data "Mancala, thereby giving you an extra move. It can also show you opportunities to"
data "capture some of your opponent's stones. You will see the same BLUE outline to show"
data "the last stone of the computer's move.
data ""
data "When stones in a player's pits are being captured by the opponent, the pit where the"
data "stones are being captured from will be outlined in RED.
data ""
data "At the conclusion of each game, the program will show a box that tells you the number"
data "of games that have been played in this session, and the number of winning games by the"
data "player, by the computer, and ties. You can then choose to play another game or quit the"
data "program. When playing multiple games, the first move alternates between you and the"
data "computer.

